/************************************************************************
 * @file: CThread.cpp
 *
 * @version: 0.1
 *
 * @description: This source file contains implementation for Base class
 * CThread to create and register a thread. CThread class will be used by
 * BackendALSA class for handling BackendALSA class worker Thread.
 *
 * @component: platform/abalsa
 *
 * @author: Jens Lorenz, jlorenz@de.adit-jv.com 2015
 *
 * @copyright (c) 2015 Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 ***********************************************************************/

#include <stdexcept>
#include <cassert>
#include "CThread.h"

using namespace adit::utility;

#define THROW_ASSERT_NEQ(CALL, COND) \
        if (!((CALL) == (COND))) throw std::runtime_error( \
            std::string(__func__) + ": (" + std::string(#CALL) + " != " + std::to_string(COND) + ")")


CThread::CThread()
    : mId(), mName(), mState(STATE_JOINED), mThreadErr(0)
{
    THROW_ASSERT_NEQ(pthread_mutex_init(&mMtx, nullptr), 0);
    THROW_ASSERT_NEQ(pthread_cond_init(&mCond, nullptr), 0);
}

CThread::~CThread()
{
    CThread::joinThread();
    pthread_cond_destroy(&mCond);
    pthread_mutex_destroy(&mMtx);
}

void CThread::setThreadName(const std::string & name)
{
    mName = name;
    mName.resize(PTHREAD_NAME_LEN);

    if (isStateForked())
    {
        // the cast inside is for Lint
        pthread_setname_np(mId, static_cast<const char*>(mName.c_str()));
    }
}

void CThread::setThreadSched(const int policy, const int priority)
{
    __cpu_sched cpuSched(policy, priority);
    setThreadSched(cpuSched);
}

void CThread::setThreadSched(const __cpu_sched & cpuSched)
{
    mCpuSched = cpuSched;

    if (isStateForked())
    {
        int policy;
        struct sched_param param;
        pthread_getschedparam(mId, &policy, &param);
        param.__sched_priority = mCpuSched.priority;
        pthread_setschedparam(mId, mCpuSched.policy, &param);
    }
}

int CThread::startThread()
{
    int ret = 0;

    switch (getState())
    {
        case STATE_JOINED:
            ret = startWorkerThread();
            break;
        case STATE_JOINING:
            waitForStateChange(STATE_JOINING);
            ret = startWorkerThread();
            break;
        case STATE_FORKING:
            waitForStateChange(STATE_FORKING);
            break;
        case STATE_STOPPING:
            waitForStateChange(STATE_STOPPING);
            setState(STATE_RUNNING);
            break;
        case STATE_STOPPED:
            setState(STATE_RUNNING);
            break;
        default:
            break;
    }

    return ret;
}

int CThread::stopThread()
{
    switch(getState())
    {
        case STATE_FORKING:
            waitForStateChange(STATE_FORKING);
            setStateAndWaitForStateChange(STATE_STOPPING);
            break;
        case STATE_RUNNING:
            setStateAndWaitForStateChange(STATE_STOPPING);
            break;
        case STATE_STOPPING:
            waitForStateChange(STATE_STOPPING);
            break;
        case STATE_STOPPED:
            break;
        case STATE_JOINING:
        case STATE_JOINED:
        default:
            return -1;
    }

    return 0;
}

int CThread::joinThread()
{
    switch (getState())
    {
        case STATE_FORKING:
            waitForStateChange(STATE_FORKING);
            setState(STATE_JOINING);
            break;
        case STATE_RUNNING:
        case STATE_STOPPING:
        case STATE_STOPPED:
            setState(STATE_JOINING);
            break;
        case STATE_JOINED:
            return mThreadErr;
        case STATE_JOINING:
        default:
            break;
    }

    pthread_join(mId, NULL);
    setState(STATE_JOINED);
    return mThreadErr;
}

int CThread::startWorkerThread()
{
    pthread_attr_t attr;

    /* initialize thread attributes */
    THROW_ASSERT_NEQ(pthread_attr_init(&attr), 0);
    THROW_ASSERT_NEQ(pthread_attr_setdetachstate(&attr, PTHREAD_CREATE_JOINABLE), 0);
    if ((mCpuSched.policy != SCHED_OTHER) || (mCpuSched.priority != 0))
    {
        struct sched_param param;
        THROW_ASSERT_NEQ(pthread_attr_getschedparam(&attr, &param), 0);
        param.sched_priority = mCpuSched.priority;
        THROW_ASSERT_NEQ(pthread_attr_setschedpolicy(&attr, mCpuSched.policy), 0);
        THROW_ASSERT_NEQ(pthread_attr_setschedparam(&attr, &param), 0);
        THROW_ASSERT_NEQ(pthread_attr_setinheritsched(&attr, PTHREAD_EXPLICIT_SCHED), 0);
    }

    /* start thread */
    setState(STATE_FORKING);
    THROW_ASSERT_NEQ(pthread_create(&mId, &attr, &CThread::_WorkerThread, this), 0);
    THROW_ASSERT_NEQ(pthread_attr_destroy(&attr), 0);

    waitForStateChange(STATE_FORKING);
    if (getState() != STATE_RUNNING)
    {
        int err = pthread_join(mId, NULL);
        if ((err != 0) && (mThreadErr == 0))
        {
            mThreadErr = err;
        }
        setState(STATE_JOINED);
        return mThreadErr;
    }

    return 0;
}

void CThread::WorkerThread()
{
    if (mName.size() != 0)
    {
        // the cast inside is for Lint
        pthread_setname_np(mId, static_cast<const char*>(mName.c_str()));
    }

    mThreadErr = static_cast<CThread*>(this)->initThread();
    if (mThreadErr == 0)
    {
        setState(STATE_RUNNING);
        do
        {
            mThreadErr = static_cast<CThread*>(this)->workerThread();
            if (getState() == STATE_STOPPING)
            {
                setStateAndWaitForStateChange(STATE_STOPPED);
            }

        } while ((mThreadErr == 0) && (isStateForked()));
    }

    setState(STATE_JOINING);
    static_cast<CThread*>(this)->deinitThread(mThreadErr);

    pthread_exit(NULL);
}

void CThread::setState(const eState state)
{
    THROW_ASSERT_NEQ(pthread_mutex_lock(&mMtx), 0);
    mState = state;
    THROW_ASSERT_NEQ(pthread_cond_signal(&mCond), 0);
    THROW_ASSERT_NEQ(pthread_mutex_unlock(&mMtx), 0);
}

CThread::eState CThread::getState()
{
    return mState;
}

bool CThread::isStateForked()
{
    return static_cast<bool>(mState & (STATE_RUNNING | STATE_STOPPING | STATE_STOPPED));
}

void CThread::waitForStateChange(const eState state)
{
    THROW_ASSERT_NEQ(pthread_mutex_lock(&mMtx), 0);
    while (mState == state)
    {
        THROW_ASSERT_NEQ(pthread_cond_wait(&mCond, &mMtx), 0);
    }
    THROW_ASSERT_NEQ(pthread_mutex_unlock(&mMtx), 0);
}

void CThread::setStateAndWaitForStateChange(const eState state)
{
    THROW_ASSERT_NEQ(pthread_mutex_lock(&mMtx), 0);
    mState = state;
    THROW_ASSERT_NEQ(pthread_cond_signal(&mCond), 0);
    while (mState == state)
    {
        THROW_ASSERT_NEQ(pthread_cond_wait(&mCond, &mMtx), 0);
    }
    THROW_ASSERT_NEQ(pthread_mutex_unlock(&mMtx), 0);
}
